import random
import time
from RPiKeyboardConfig import RPiKeyboardConfig

keyboard = RPiKeyboardConfig()

# Build a grid from the matrix info
all_leds = keyboard.get_leds()
positions = [led.matrix for led in all_leds]
pos_to_idx = {tuple(led.matrix): led.idx for led in all_leds}

def random_board():
    return {(r, c): random.choice([0, 1]) for r, c in positions}

def count_neighbors(board, r, c):
    neighbors = 0
    for dr in (-1, 0, 1):
        for dc in (-1, 0, 1):
            if dr == 0 and dc == 0:
                continue
            if (r+dr, c+dc) in board:
                neighbors += board[(r+dr, c+dc)]
    return neighbors

def next_gen(board):
    new_board = {}
    for r, c in board.keys():
        n = count_neighbors(board, r, c)
        if board[(r, c)] == 1:
            new_board[(r, c)] = 1 if n in (2, 3) else 0
        else:
            new_board[(r, c)] = 1 if n == 3 else 0
    return new_board

def draw_board(board, alive_colour):
    for (r, c), alive in board.items():
        idx = pos_to_idx[(r, c)]
        if alive:
            keyboard.set_led_by_idx(idx=idx, colour=alive_colour)
        else:
            keyboard.set_led_by_idx(idx=idx, colour=(0, 0, 0))
    keyboard.send_leds()

def run_game(fps=10, max_generations=200, linger_frames=4):
    keyboard.set_led_direct_effect()

    board = random_board()
    alive_colour = (random.randint(0, 255), 255, 255)  # Random hue each run
    history = []
    gen = 0
    linger = 0

    while True:
        draw_board(board, alive_colour)
        time.sleep(1/fps)

        new_board = next_gen(board)
        gen += 1

        # Track history for loops/static boards
        history.append(new_board)
        if len(history) > 30:
            history.pop(0)

        # Reset conditions
        dead = all(v == 0 for v in new_board.values())
        looping = new_board in history[:-1]
        over_gen = gen > max_generations

        if dead or looping or over_gen:
            linger += 1
            if linger >= linger_frames:
                print("Resetting board...")
                board = random_board()
                alive_colour = (random.randint(0, 255), 255, 255)  # New hue
                history.clear()
                gen = 0
                linger = 0
                continue

        board = new_board

if __name__ == "__main__":
    run_game(fps=5)  # you can crank this to 30fps
